// Copyright (C) 1999-2005 Siebel Systems, inc.  All rights reserved.
function TableMatrix(domTbl)
{
   this.domTbl = domTbl;
   
   this.reset();
}

TableMatrix.generateAnnotations = function(evt, objId)
{
   var table = saw.getEventTarget(evt);
   if(table.nodeName != "TABLE" || table.annotated)
      return false;
   
   table.annotated = true;
   var tm = new TableMatrix(table);
   tm.annotate(objId);
}

TableMatrix.isRowTotalHeaderCell = function(cell)
{
   var className = cell.className;
   return (cell.colspan > 1 || (className.match(/\bPTROW\d+T\b/) != null));
}

TableMatrix.isCellEmpty = function(cell)
{
   return (cell.firstChild == null);
}

TableMatrix.prototype.generateCellId = function(cell, r, c)
{
   if(!cell.id)
      cell.id = this.domTbl.id + "_" + r + "_" + c;
}

TableMatrix.prototype.isMeasureCell = function(r, c)
{
   var tableObj = this.tableObj;
   if(tableObj.yLayers)
   {
      var colPos = this.getColumnPosition(r, c);
      return (colPos >= tableObj.yLayers.length);
   }
   else
      return tableObj.vColumnAggs[tableObj.vRSColumns[c]];
}


TableMatrix.prototype.annotate = function(objId)
{
   this.id = this.domTbl.id;
   this.tableObj = window[objId];
   var nXLayers = this.tableObj.xLayers ? this.tableObj.xLayers.length : 1;
   
   var rows = this.domTbl.rows;
   var bSummarized = false;
   for(var i = 0; i < rows.length; ++i)
   {
      if(i < nXLayers)
         this.annotateColHeaderRow(i);
      else
      {
         if(!bSummarized)
         {
            bSummarized = true;
            this.summarizeColHeaders(i);
         }
         this.annotateRow(i);
      }
   }
}

TableMatrix.prototype.summarizeColHeaders = function(r) // first r rows
{
   this.colHeaders = new Array();
   for(var c = 0; c < this.nColumns; ++c)
   {
      var colCells = this.getUniqueColumnCells(c, r);
      var lastCell = colCells[colCells.length - 1];
      this.colHeaders[c] = lastCell.headers + " " + lastCell.id;
   }
}

TableMatrix.prototype.annotateColHeaderRow = function(r)
{
   var cells = this.getUniqueRowCells(r);
   for(var c = 0; c < cells.length; ++c)
   {
      var cell = cells[c];
      if(TableMatrix.isCellEmpty(cell) || cell.id)
         continue;
         
      this.generateCellId(cell, r, c);
      cell.headers = "";
      
      var colPos = this.getColumnPosition(r, c);
      var colCells = this.getUniqueColumnCells(colPos, r);
      for(var j = 0; j < colCells.length; ++j)
      {
         if(colCells[j].id)
            cell.headers += colCells[j].id + " ";
      }
         
      if(cell.headers.length > 0)
         cell.headers = cell.headers.substr(0, cell.headers.length - 1);
   }
}

TableMatrix.prototype.annotateRow = function(r)
{
   var cells = this.getUniqueRowCells(r);
   for(var c = 0; c < cells.length; ++c)
   {
      if(this.isMeasureCell(r, c))
         this.annotateMeasureCell(r, c);
      else
      {
         this.generateCellId(cells[c], r, c);
         
         if(TableMatrix.isRowTotalHeaderCell(cells[c]))
            this.annotateRowTotalHeaderCell(r, c);
         else
            this.annotateMeasureCell(r, c);
      }
   }
}

TableMatrix.prototype.annotateMeasureCell = function(r, c)
{
   var colPos = this.getColumnPosition(r, c);
   var rowCells = this.getUniqueRowCells(r);
   var cell = rowCells[c];
   cell.headers = this.colHeaders[colPos];
   for(var i = 0; i < c; ++i)
   {
      if(rowCells[i].id)
         cell.headers += " " + rowCells[i].id;
   }
}

TableMatrix.prototype.annotateRowTotalHeaderCell = function(r, c)
{
   var rowCells = this.getUniqueRowCells(r);
   var cell = rowCells[c];
   cell.headers = "";
   for(var i = 0; i < c; ++i)
   {
      if(rowCells[i].id)
         cell.headers += rowCells[i].id + " ";
   }
   
   if(cell.headers.length > 0)
      cell.headers = cell.headers.substr(0, cell.headers.length - 1);
}

// reset matrix (optionally to point at a new table)
TableMatrix.prototype.reset = function(domTbl)
{
   if(domTbl != null)
      this.domTbl = domTbl;
      
   this.rowArray = new Array;
   this.nColumns = 0;
   this.rowArray[0] = new Array;
   this.rowArray[0].bComplete = true;
   this.rowArray[0].nCell = 0;

   for(var c = 0 ; c != this.domTbl.rows[0].cells.length ; ++c)
   {
      this.nColumns += this.domTbl.rows[0].cells[c].colSpan;
   }
}

// get cells belonging to first nRows rows of column c (some cells may be non-unique)
TableMatrix.prototype.getColumnCells = function(c, nRows)
{
   var a = new Array();

   if(nRows == null)
      nRows = this.domTbl.rows.length;
      
   while(nRows > 0)
   {
      a.push(this.getCell(--nRows,c));
   }

   a.reverse();

   return a;
}

// get cells belonging to first nCols columns of row r (some cells may be non-unique)
TableMatrix.prototype.getRowCells = function(r, nCols)
{
   var a = new Array();

   if(nCols == null)
      nCols = this.nColumns;

   // faster to do backwards (and then reverse at end)      
   while(nCols > 0)
   {
      a.push(this.getCell(r,--nCols));
   }

   a.reverse();

   return a;
}

// returns only the unique cells along a column
TableMatrix.prototype.getUniqueColumnCells = function(c, nRows)
{
   var a = new Array();

   if(nRows == null)
      nRows = this.domTbl.rows.length;

   var r = 0;
   
   while(r < nRows)
   {
      var cur = this.getCell(r,c);
      
      r += cur.rowSpan;

      a.push(cur);
   }
   
   return a;
}

// returns only the unique cells along a row
TableMatrix.prototype.getUniqueRowCells = function(r, nCols)
{
   var a = new Array();
   
   if(nCols == null)
      nCols = this.nColumns;
    
   var c = 0;
   
   while(c < nCols)
   {
      var cur = this.getCell(r,c);
      
      c += cur.colSpan;
      
      a.push(cur);
   }
  
   return a;
} 
   
// returns the column index of a cell identified by a rowIndex and cellIndex 
TableMatrix.prototype.getColumnPosition = function(rowIndex, cellIndex)
{
   var nPos = 0;

   if(this.rowArray[rowIndex] && this.rowArray[rowIndex].bComplete)
   {
      while(cellIndex-- > 0)
      {
         nPos += this.domTbl.rows[rowIndex].cells[cellIndex].colSpan;
      }

      return nPos;
   }

   var tRowCells = this.getRowCells(rowIndex, this.nColumns);
   var tTargetCell = this.getUniqueRowCells(rowIndex)[cellIndex];
   var i = 0;
   
   while(tRowCells[i] != tTargetCell)
      ++i;
   
   return i;
}

TableMatrix.prototype.getCell = function(r,c)
{
   if(this.rowArray[r])
   {
      if(this.rowArray[r][c])
         return this.rowArray[r][c];
   }
   else
   {
      this.rowArray[r] = new Array();

      this.rowArray[r].bComplete = this.isRowComplete(r);
      this.rowArray[r].nCell = 0;
   }

   if(this.rowArray[r].bComplete)
   {
      var nCell = 0;

      while(c >= this.domTbl.rows[r].cells[nCell].colSpan)
      {
         c -= this.domTbl.rows[r].cells[nCell].colSpan;

         ++nCell;
      }

      return this.domTbl.rows[r].cells[nCell];
   }

   var completeRow = r;

   do
   {
      --completeRow;

      if(!this.rowArray[completeRow])
      {
         this.rowArray[completeRow] = new Array;
         this.rowArray[completeRow].bComplete = this.isRowComplete(completeRow);
         this.rowArray[completeRow].nCell = 0;
      }
   } while(!this.rowArray[completeRow].bComplete);

   for(ar = completeRow; ar <= r ; ++ar)
   {
      var arrayRow = this.rowArray[ar];

      var domRow = this.domTbl.rows[ar];

      var ac = 0;

      while(ac <= c && this.rowArray[ar].nCell < domRow.cells.length)
      {
         var domCell = domRow.cells[this.rowArray[ar].nCell];

         while(arrayRow[ac])
            ++ac;

         if(ac > c)
            break;

      //   alert('filling cell(' + nCell + '): '+ar+','+ac + ' ' + domCell.innerText);

         for(var rr = 0; rr != domCell.rowSpan ; ++rr)
         {
            var rrr = this.rowArray[rr+ar];

            if(!rrr)
            {
               this.rowArray[rr+ar] = rrr = new Array;
               rrr.bComplete = false;
               rrr.nCell = 0;
            }

            for(var cc = 0 ; cc != domCell.colSpan ; ++cc)
            {
               rrr[cc+ac] = domCell;
            }
         }

        var bFoundIt = (c >= ac && c < ac+domCell.colSpan && r >= ar && r < ar+domCell.rowSpan);

         ac += domCell.colSpan;
         ++this.rowArray[ar].nCell;

         if(bFoundIt)
            return this.rowArray[r][c];
      }

   }

   return this.rowArray[r][c];
}

// internal
TableMatrix.prototype.isRowComplete = function(r)
{
   var nC = 0;

   for(var c = 0 ; c < this.domTbl.rows[r].cells.length ; ++c)
   {
      nC += this.domTbl.rows[r].cells[c].colSpan;
   }

   return nC == this.nColumns;
}
